unit sablotengine;
{
/*
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 * The Original Code is the Sablopas : Pascal wrapper for Sablotron.
 * More infos at http://www.tekool.com/sablopas - Contact : cespern@free.fr
 *
 * The Initial Developer of the Original Code is Christophe ESPERN
 * Portions created by Christophe ESPERN are Copyright (C) 2001-2002 Christophe
 * ESPERN. All Rights Reserved.
 *
 * Contributor(s): None
}

interface

{$IFDEF FPC}
 {$MODE DELPHI}
{$ENDIF}

uses
  SysUtils,Classes,shandler,sablot,sabutils;

const

  XML_HEADER='<?xml version="1.0"?>';
  XSL_HEADER= '<?xml version="1.0"?>'+
              '<xsl:stylesheet version="1.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform"'
              +' xmlns:xlink="http://www.w3.org/1999/xlink" >'
              +'</xsl:stylesheet>';

 XSL_FILE=0;
 XML_FILE=1;

type

  TSablotEngine=class;

  {Error Event}
  TSErrorEvent = procedure (Sender:TObject;const ErrorCode:integer;var 
          Fields:TSErrorFields) of object;


 {Scheme Events}
  TSFreeMemoryEvent = procedure (Sender:TObject;buffer:Pchar) of object;
  TSGetAllEvent = procedure (Sender:TObject;const scheme,rest:PChar;var 
          buffer:Pchar; var bytecount:integer) of object;
  TSCloseEvent = procedure (Sender:TObject;const handle:integer) of object;
  TSGetEvent = procedure (Sender:TObject;const handle:integer;var buffer:Pchar; 
          var bytecount:integer) of object;
  TSPutEvent = procedure (Sender:TObject;const handle:integer;const 
          buffer:PChar;var byteCount:integer) of object;
  TSOpenEvent = procedure (Sender:TObject;const scheme,rest:Pchar;var 
          handle:integer) of object;
  TSHasHandleEvent = procedure (Sender:TObject;handle:integer;var 
          Result:boolean) of object;
  {CustomSchemeProxy}

  TCustomSchemeProxy = class (TComponent)
  private
    FEngine: TSablotEngine;
    FSchemes: TStringlist;
    procedure SetEngine(Value: TSablotEngine);
  protected
    function Close(handle:integer): Integer; virtual; abstract;
    function FreeMemory(var buffer:PChar): Integer; virtual; abstract;
    function Get(const handle:integer;var buffer:PChar;var bytecount:integer): 
            Integer; virtual; abstract;
    function GetAll(const scheme,rest:Pchar;var buffer:Pchar;var 
            bytecount:integer): Integer; virtual; abstract;
    function HasHandle(handle:integer): Boolean; virtual; abstract;
    procedure Notification(AComponent: TComponent; Operation: TOperation); 
            override;
    function Open(const scheme,rest:Pchar;var handle:integer): Integer; virtual;
            abstract;
    function Put(handle:integer;const buffer:PChar;var byteCount:integer): 
            Integer; virtual; abstract;
  public
    constructor Create(Aowner:TComponent); override;
    destructor Destroy; override;
    procedure AddScheme(AScheme:string);
    function HasScheme(AScheme:string): Boolean;
    procedure RemoveScheme(AScheme:string);
  published
    property Engine: TSablotEngine read FEngine write SetEngine;
  end;

  {FileProxy}

  TFileProxy = class (TCustomSchemeProxy)
  private
    FHandles: TList;
    FInBuff: string;
    FChanged:boolean;
    FOpenMode:word;
  protected
    function Close(handle:integer): Integer; override;
    function FreeMemory(var buffer:PChar): Integer; override;
    function Get(const handle:integer;var buffer:PChar;var bytecount:integer): 
            Integer; override;
    function GetAll(const scheme,rest:Pchar;var buffer:Pchar;var 
            bytecount:integer): Integer; override;
    function HasHandle(handle:integer): Boolean; override;
    function Open(const scheme,rest:Pchar;var handle:integer): Integer;
            override;
    function Put(handle:integer;const buffer:PChar;var byteCount:integer): 
            Integer; override;
  public
    constructor Create(Aowner:TComponent); override;
    destructor Destroy; override;
  end;
  

  {SchemeProxy}
  TSchemeProxy = class (TCustomSchemeProxy)
  private
    FOnClose: TSCloseEvent;
    FOnFreeMemory: TSFreeMemoryEvent;
    FOnGet: TSGetEvent;
    FOnGetAll: TSGetAllEvent;
    FOnHasHandle: TSHasHandleEvent;
    FOnOpen: TSOpenEvent;
    FOnPut: TSPutEvent;
  protected
    function Close(handle:integer): Integer; override;
    function FreeMemory(var buffer:PChar): Integer; override;
    function Get(const handle:integer;var buffer:PChar;var bytecount:integer): 
            Integer; override;
    function GetAll(const scheme,rest:Pchar;var buffer:Pchar;var 
            bytecount:integer): Integer; override;
    function HasHandle(handle:integer): Boolean; override;
    function Open(const scheme,rest:Pchar;var handle:integer): Integer; 
            override;
    function Put(handle:integer;const buffer:PChar;var byteCount:integer): 
            Integer; override;
  published
    property OnClose: TSCloseEvent read FOnClose write FOnClose;
    property OnFreeMemory: TSFreeMemoryEvent read FOnFreeMemory write 
            FOnFreeMemory;
    property OnGet: TSGetEvent read FOnGet write FOnGet;
    property OnGetAll: TSGetAllEvent read FOnGetAll write FOnGetAll;
    property OnHasHandle: TSHasHandleEvent read FOnHasHandle write FOnHasHandle;
    property OnOpen: TSOpenEvent read FOnOpen write FOnOpen;
    property OnPut: TSPutEvent read FOnPut write FOnPut;
  end;
  
  {SablotEngine}
  TSablotEngine = class (TComponent)
  private
    FBases: TStringlist;
    FDefaultBase: string;
    FFileProxy: TFileProxy;
    FInputUri: string;
    FLastDebug: string;
    FLastError: string;
    FMsgHandler: SMessageHandler;
    FOnMsgCritical: TSErrorEvent;
    FOnMsgDebug: TSErrorEvent;
    FOnMsgError: TSErrorEvent;
    FOnMsgInfo: TSErrorEvent;
    FOnMsgWarning: TSErrorEvent;
    FOutputUri: string;
    FParams: TStringList;
    FProcessor: PProcessor;
    FProxies: TList;
    FSchemeHandler: SSchemeHandler;
    FShowDebug: Boolean;
    FSituation: PSituation;
    FStyleURI: string;
    FTempProxy: TCustomSchemeProxy;
    function GetBaseByScheme(Scheme:string): string;
    function GetBaseCount: Integer;
    function GetProxies(index:integer): TCustomSchemeProxy;
    function GetProxyByHandle(AHandle:integer): TCustomSchemeProxy;
    function GetProxyCount: Integer;
    procedure InitSchemeHandler;

  protected
      procedure InternalClearBases;
    procedure PrepareBases;
    procedure PrepareParams;
    procedure AddProxy(AProxy:TCustomSchemeProxy);
    procedure ClearBases;
    procedure ClearError;
    procedure ClearParams;
    procedure DoRaiseEvent(code:MH_ERROR;level:MH_LEVEL;var 
            fields:TSErrorFields); virtual;
    function GetDefaultBase: string;
    procedure RemoveProxy(AProxy:TCustomSchemeProxy);
    function SabCreatePDocument(d:PDomDocument): Integer;
    function SabDestroyPDocument(d:SDomDocument): Integer;
    function SabLockDocument(d:SDomDocument): Integer;
    function SabParse(URI:string;d:PDomDocument): Integer;
    function SabParseBuffer(const buffer:string;d:PDomDocument): Integer;
    function SabParseStylesheet(URI:string;d:PDomDocument): Integer;
    function SabParseStylesheetBuffer(const buffer:string;d:PDomDocument): 
            Integer;
    procedure SetDefaultBase(const ABase: string);
    property FileProxy: TFileProxy read FFileProxy;
    property Proxies[index:integer]: TCustomSchemeProxy read GetProxies;
    property ProxyCount: Integer read GetProxyCount;
  public
    constructor Create(AOwner:TComponent); override;
    destructor Destroy; override;
    function AddArgBuffer(const argName,bufferValue:string): Boolean;
    procedure FillBaseList(strings:TStrings);
    procedure FillParamList(Strings:TStrings);
    procedure GetBasesList(strings:TStrings);
    procedure GetParamList(Strings:TStrings);
    function GetResultArg(argURI:string): string;
    function HasParam(paramName:string): Boolean;
    procedure RaiseMessage(level:MH_LEVEL;ModuleName,Msg: string); virtual;
    procedure RemoveBase(const scheme:string); virtual;
    procedure RemoveParam(const paramName:string); virtual;
    procedure ResolveSabURI(var Scheme,Rest:string);
    function Run: Boolean; virtual;
    function RunProcessorGen(AStyleURI,AInputURI,AOutputURI:string): Boolean;
    procedure SetBase(const scheme,base:string);
    procedure SetParam(const paramName,paramValue:string);
    property BaseByScheme[Scheme:string]: string read GetBaseByScheme;
    property BaseCount: Integer read GetBaseCount;
    property LastDebug: string read FLastDebug;
    property LastError: string read FLastError;
    property Processor: PProcessor read FProcessor;
    property Situation: PSituation read FSituation;
  published
    property DefaultBase: string read GetDefaultBase write SetDefaultBase;
    property InputUri: string read FInputUri write FInputUri;
    property OnMsgCritical: TSErrorEvent read FOnMsgCritical write 
            FOnMsgCritical;
    property OnMsgDebug: TSErrorEvent read FOnMsgDebug write FOnMsgDebug;
    property OnMsgError: TSErrorEvent read FOnMsgError write FOnMsgError;
    property OnMsgInfo: TSErrorEvent read FOnMsgInfo write FOnMsgInfo;
    property OnMsgWarning: TSErrorEvent read FOnMsgWarning write FOnMsgWarning;
    property OutputUri: string read FOutputUri write FOutputUri;
    property ShowDebug: Boolean read FShowDebug write FShowDebug;
    property StyleURI: string read FStyleURI write FStyleURI;
  end;
  
procedure Register;


implementation

{$I handlers.inc}


procedure Register;
begin
  RegisterComponents('Tekool', [TSablotEngine]);
  RegisterComponents('Tekool', [TSchemeProxy]);
end;

{ TSchemeProxy }
{
****************************** TCustomSchemeProxy ******************************
}
constructor TCustomSchemeProxy.Create(Aowner:TComponent);
begin
  inherited;
  FSchemes:=TStringList.create;
  FSchemes.Duplicates:=dupError;
end;

destructor TCustomSchemeProxy.Destroy;
begin
  SetEngine(nil);
  FSchemes.free;
  inherited;
end;

procedure TCustomSchemeProxy.AddScheme(AScheme:string);
begin
  if FSchemes.IndexOf(AScheme)=-1 then
     FSchemes.Add(AScheme)
end;

function TCustomSchemeProxy.HasScheme(AScheme:string): Boolean;
begin
  if FSchemes.IndexOf(AScheme)<>-1 then
    Result:=true
  else
    Result:=false;
end;

procedure TCustomSchemeProxy.Notification(AComponent: TComponent; Operation: 
        TOperation);
begin
  if (AComponent=FEngine) and (Operation=opRemove) then
   SetEngine(nil);
  inherited;
end;

procedure TCustomSchemeProxy.RemoveScheme(AScheme:string);
begin
  if FSchemes.IndexOf(AScheme)<>-1 then
   FSchemes.Delete(FSchemes.IndexOf(AScheme));
end;

procedure TCustomSchemeProxy.SetEngine(Value: TSablotEngine);
begin
  if FEngine<>value then begin
   if FEngine<>nil then
     FEngine.RemoveProxy(self);
    FEngine := Value;
   if FEngine<>nil then
     FEngine.AddProxy(self)
  end
end;

{
********************************* TSchemeProxy *********************************
}
function TSchemeProxy.Close(handle:integer): Integer;
begin
  if Assigned(FOnClose) then
   FOnClose(self,handle);
   Result:=0;
end;

function TSchemeProxy.FreeMemory(var buffer:PChar): Integer;
begin
  Result:=1;
  if Assigned(FOnFreeMemory) then
   FOnFreeMemory(self,buffer);
  if buffer=nil then
   Result:=0
end;

function TSchemeProxy.Get(const handle:integer;var buffer:PChar;var 
        bytecount:integer): Integer;
begin
  Result:=1;
  if Assigned(FOnGet) then
   FOnGet(self,handle,buffer,bytecount);
  if bytecount<>0 then
   Result:=0;
end;

function TSchemeProxy.GetAll(const scheme,rest:Pchar;var buffer:Pchar;var 
        bytecount:integer): Integer;
begin
  if Assigned(FOnGetAll) then
    FOnGetAll(self,scheme,rest,buffer,bytecount);
  if bytecount=0 then
   Result:=1
  else
   Result:=0;
end;

function TSchemeProxy.HasHandle(handle:integer): Boolean;
begin
  Result:=false;
  if Assigned(FOnHasHandle) then
   FOnHasHandle(self,handle,result);
end;

function TSchemeProxy.Open(const scheme,rest:Pchar;var handle:integer): Integer;
begin
  Result:=1;
  if Assigned(FOnOpen) then
    FOnOpen(self,scheme,rest,handle);
  if handle<>0 then
   result:=0;
end;

function TSchemeProxy.Put(handle:integer;const buffer:PChar;var 
        byteCount:integer): Integer;
begin
  Result:=1;
  if Assigned(FOnPut) then begin
   FOnPut(self,handle,buffer,bytecount);
   if bytecount=0 then
    Result:=1
   else
    Result:=0;
  end
end;

{
******************************** TSablotEngine *********************************
}
constructor TSablotEngine.Create(AOwner:TComponent);
begin
  inherited;
  FProxies:=TList.create;
  FFileProxy:=TFileProxy.Create(self);
  FFileProxy.engine:=self;
  FParams:=TStringlist.create;
  FBases:=TStringlist.create;
  FShowDebug:=false;
  //Create SablotSituation and SablotProcessor
  GetMem(FProcessor,SizeOf(PSablotHandle));
  GetMem(FSituation,SizeOf(PSablotHandle));
  SablotCreateSituation(FSituation);
  SablotCreateProcessorForSituation(FSituation^,FProcessor);
  //Init and record the Message Handler
  FMsgHandler.makeCode:=@SablotEngine.MessageHandlerMakeCode;
  FMsgHandler.log:=@SablotEngine.MessageHandlerLog;
  FMsgHandler.error:=@SablotEngine.MessageHandlerError;
  SablotRegHandler(FProcessor^,HLR_MESSAGE,@FMsgHandler,self);
  InitSchemeHandler;
end;

destructor TSablotEngine.Destroy;
begin
  SablotUnRegHandler(FProcessor^,HLR_MESSAGE,@FMsgHandler,self);
  SablotUnRegHandler(FProcessor^,HLR_SCHEME,@FSchemeHandler,self);
  SablotDestroyProcessor(FProcessor^);
  SablotDestroySituation(FSituation^);
  FreeMem(FProcessor,SizeOf(PSablotHandle));
  FreeMem(FSituation,SizeOf(PSablotHandle));
  inherited;
  FParams.free;
  FBases.free;
  FProxies.free;
end;

function TSablotEngine.AddArgBuffer(const argName,bufferValue:string): Boolean;
begin
  try
   if SablotAddArgBuffer(FSituation^,FProcessor^,PChar(ArgName),PCHar(BufferValue))=0 then
    Result:=true
   else
    Result:=false;
  except
   raise Exception.Create('TSablotEngine.AddArgBuffer');
  end;
end;

procedure TSablotEngine.AddProxy(AProxy:TCustomSchemeProxy);
begin
  FProxies.Add(AProxy);
end;

procedure TSablotEngine.ClearBases;
begin
  InternalClearBases;
  FBases.clear;
end;

procedure TSablotEngine.ClearError;
begin
  try
   SablotClearError(Fprocessor^);
   SablotClearSituation(FSituation^);
  except
   raise Exception.Create('TSablotEngine.Clear Error');
  end;
end;

procedure TSablotEngine.ClearParams;
begin
  FParams.Clear;
end;

procedure TSablotEngine.DoRaiseEvent(code:MH_ERROR;level:MH_LEVEL;var 
        fields:TSErrorFields);
begin
  case level of
   MH_LEVEL_DEBUG:if Assigned(FOnMsgDebug)then
     FOnMsgDebug(self,code,fields);
   MH_LEVEL_INFO:if Assigned(FOnMsgInfo)then
     FOnMsgInfo(self,code,fields);
   MH_LEVEL_WARN:if Assigned(FOnMsgWarning)then
     FOnMsgWarning(self,code,fields);
   MH_LEVEL_ERROR:if Assigned(FOnMsgError)then
     FOnMsgError(self,code,fields);
   MH_LEVEL_CRITICAL:if Assigned(FOnMsgCritical)then
     FOnMsgCritical(self,code,fields);
  end;
end;

procedure TSablotEngine.FillBaseList(strings:TStrings);
begin
  FBases.Assign(strings);
end;

procedure TSablotEngine.FillParamList(Strings:TStrings);
begin
  FParams.Assign(strings);
end;

function TSablotEngine.GetBaseByScheme(Scheme:string): string;
var
  i: Integer;
  s1, s2: string;
begin
  for i:=0 to FBases.count-1 do begin
   SplitKeyValue(FBases[i],s1,s2);
   if s1=scheme then begin
     Result:=s2;
     exit;
   end;
  end;
end;

function TSablotEngine.GetBaseCount: Integer;
begin
  Result:=FBases.Count;
end;

procedure TSablotEngine.GetBasesList(strings:TStrings);
begin
  Strings.Assign(FBases);
end;

function TSablotEngine.GetDefaultBase: string;
begin
  Result:=FDefaultbase;
end;

procedure TSablotEngine.GetParamList(Strings:TStrings);
begin
  Strings.Assign(FParams);
end;

function TSablotEngine.GetProxies(index:integer): TCustomSchemeProxy;
begin
  if (index>-1) and (index<FProxies.count) then
   Result:= FProxies[index]
  else
   Result:=nil;
end;

function TSablotEngine.GetProxyByHandle(AHandle:integer): TCustomSchemeProxy;
var
  i: Integer;
begin
  Result:=nil;
  if  FileProxy.HasHandle(Ahandle) then
   Result:=FileProxy
  else
  with FProxies do
   for i:=0 to count-1 do
    if Proxies[i].HAsHandle(AHandle) then begin
     Result:=Proxies[i];
     break;
    end;
end;

function TSablotEngine.GetProxyCount: Integer;
begin
  Result:=FProxies.Count;
end;

function TSablotEngine.GetResultArg(argURI:string): string;
var
  tmp: PChar;
begin
  tmp:='';
  try
   SablotGetResultArg(Fprocessor^,Pchar(argUri),@tmp);
   Result:=string(tmp);
   SablotFree(tmp);
  except
   RaiseMessage(MH_LEVEL_DEBUG,'Sablopas','try to get an unknown output ?');
  end;
end;

function TSablotEngine.HasParam(paramName:string): Boolean;
var
  i: Integer;
  s1, s2: string;
begin
  Result:=false;
  for i:=0 to FParams.count-1 do begin
   SplitKeyValue(FParams[i],s1,s2);
   if s1=paramName then begin
    Result:=true;
     break;
   end;
  end;
end;

procedure TSablotEngine.InitSchemeHandler;
begin
  try
   FSchemeHandler.put:=@SablotEngine.SabSchemeHandlerPut;
   FSchemeHandler.get:=@SablotEngine.SabSchemeHandlerGet;
   FSchemeHandler.open:=@SablotEngine.SabSchemeHandlerOpen;
   FSchemeHandler.close:=@SablotEngine.SabSchemeHandlerClose;
   FSchemeHandler.getAll:=@SablotEngine.SabSchemeHandlerGetAll;
   FSchemeHandler.freeMemory:=@SablotEngine.SabSchemeHandlerFreeMemory;
   SablotRegHandler(FProcessor^,HLR_Scheme,@FSchemeHandler,self);
  except
   raise Exception.Create('TSablotEngine.InitSchemeHandler');
  end;
end;

procedure TSablotEngine.InternalClearBases;
var
  i: Integer;
  s1, s2: string;
begin
  try
   for i:=0 to FBases.count-1 do begin
    splitKeyValue(FBases[i],s1,s2);
    s2:='';
    if SablotSetBaseForScheme(FProcessor^,PCHar(s1),PCHar(s2))<>0 then
     RaiseMessage(MH_LEVEL_ERROR,'Sablopas','Cannot clear base for '+S1+' : '+S2);
   end;
  except
   raise Exception.Create('TSablotEngine.SetBaseForScheme error');
  end;
end;

procedure TSablotEngine.PrepareBases;
var
  i: Integer;
  s1, s2: string;
begin
  try
   for i:=0 to FBases.count-1 do begin
    splitKeyValue(FBases[i],s1,s2);
    if SablotSetBaseForScheme(FProcessor^,PCHar(s1),PCHar(s2))<>0 then
     RaiseMessage(MH_LEVEL_ERROR,'Sablopas','Cannot set scheme base for '+S1+' : '+S2);
   end;
  except
   raise Exception.Create('TSablotEngine.PrepareBases error');
  end;
end;

procedure TSablotEngine.PrepareParams;
var
  i: Integer;
  s1, s2: string;
begin
  for i:=0 to FParams.count-1 do begin
   splitKeyValue(FParams[i],s1,s2);
   try
   if SablotAddParam(FSituation^,FProcessor^,PChar(s1),PCHar(s2))<>0 then
    RaiseMessage(MH_LEVEL_ERROR,'Sablopas','Cannot add param '+s1);
   except
    Exception.create('TSablotEngine.PrepareParams with '+s1+'='+s2);
  end;
  end;
end;

procedure TSablotEngine.RaiseMessage(level:MH_LEVEL;ModuleName,Msg: string);
var
  Fields: TSErrorFields;
begin
  
  Fields[0]:=PChar('module:'+ModuleName);
  Fields[2]:=PChar('code: 100' );
  Fields[3]:=Pchar('msg: '+Msg);
  case level of
   MH_LEVEL_DEBUG: Fields[1]:='msgtype: debug';
   MH_LEVEL_INFO: Fields[1]:='msgtype: info';
   MH_LEVEL_WARN:Fields[1]:='msgtype: warning';
   MH_LEVEL_ERROR:Fields[1]:='msgtype: error';
   MH_LEVEL_CRITICAL:Fields[1]:='msgtype: critical';
  end;
  Fields[4]:=nil;
  MessageHandlerError(self,self.FProcessor^,100,level,fields);
end;

procedure TSablotEngine.RemoveBase(const scheme:string);
var
  i: Integer;
  s1, s2: string;
begin
  for i:=0 to FBases.count-1 do begin
   SplitKeyValue(FBases[i],s1,s2);
   if s1=scheme then begin
     FBases.delete(i);
     if SablotSetBaseForScheme(FProcessor^,PCHar(s1),'')<>0 then
       Exception.create('Cannot remove base for '+S1+' : '+S2);
     break;
   end;
  end;
end;

procedure TSablotEngine.RemoveParam(const paramName:string);
var
  i: Integer;
  s1, s2: string;
begin
  for i:=0 to FParams.count-1 do begin
   SplitKeyValue(FParams[i],s1,s2);
   if s1=paramName then begin
     Fparams.delete(i);
     break;
   end;
  end;
end;

procedure TSablotEngine.RemoveProxy(AProxy:TCustomSchemeProxy);
begin
  if  FProxies.IndexOf(AProxy)<>-1 then
   FProxies.Remove(AProxy)
end;

procedure TSablotEngine.ResolveSabURI(var Scheme,Rest:string);
var
  i: Integer;
  s1, s2, authority, query, fragment, uri: string;
  filename: tfilename;
begin
  for i:=0 to FBases.count-1 do begin
   SplitKeyValue(FBases[i],s1,s2);
   if s1=scheme then begin
       //Change the scheme
     s1:=Copy(s2,0,pos('://',s2)-1);
     //Get the filename
     if s1='file' then
       UriStrToFilename(s1+':'+rest,filename,authority,query,fragment) // Conversion localfile
     else
      filename:=copy(rest,pos('//',rest)+2,length(rest)); //Enlve le double /
     filename:=DosPAthToUnixPath(filename);
     //URI Creation
     ResolveRelativeURIStr(s2,authority+'/'+filename,uri);
     if Copy(uri,length(uri),1)='/' then
      uri:=Copy(uri,0,length(uri)-1);
     //URI Split
     Rest:=Copy(uri,pos('://',uri),length(uri));
     Scheme:=s1;
     break;
   end;
  end;
end;

function TSablotEngine.Run: Boolean;
begin
  Result:= RunProcessorGen(FstyleUri,FInputUri,FOutputUri);
end;

function TSablotEngine.RunProcessorGen(AStyleURI,AInputURI,AOutputURI:string): 
        Boolean;
begin
  try
    PrepareBases;
    PrepareParams;
  
    if (AStyleURI='') or (AInputURI='') or (AOutputURI='') then
     raise Exception.Create('RunProcessorGen : Missing file argument');
    if SablotRunProcessorGen(FSituation^,FProcessor^,Pchar(AStyleUri),PCHar(AInputUri),PChar(AOutputUri))=0 then
      Result:=true
    else begin
      ClearError;
      SablotFreeResultArgs(FProcessor^);
      Result:=false;
    end;
  
    InternalClearBases;
  except
   on E:Exception do begin
    RaiseMessage(MH_LEVEL_ERROR,'Sablopas',(E.Message));
    Result:=false;
    InternalClearBases;
   end;
  end;
  
end;

function TSablotEngine.SabCreatePDocument(d:PDomDocument): Integer;
begin
  try
   Result:=SablotCreateDocument(FSituation^,d);
  except
    raise Exception.Create('TSablotEngine.DestroyPDocument');
  end;
end;

function TSablotEngine.SabDestroyPDocument(d:SDomDocument): Integer;
begin
  try
   Result:=SablotDestroyDocument(FSituation^,d)
  except
    raise Exception.Create('TSablotEngine.DestroyPDocument');
  end;
end;

function TSablotEngine.SabLockDocument(d:SDomDocument): Integer;
begin
  Result:=SablotLockDocument(FSituation^,d);
end;

function TSablotEngine.SabParse(URI:string;d:PDomDocument): Integer;
begin
  Result:=SablotParse(FSituation^,PCHar(URI),d);
end;

function TSablotEngine.SabParseBuffer(const buffer:string;d:PDomDocument): 
        Integer;
begin
  Result:=SablotParseBuffer(FSituation^,PChar(buffer),d);
end;

function TSablotEngine.SabParseStylesheet(URI:string;d:PDomDocument): Integer;
begin
  Result:=SablotParseStyleSheet(FSituation^,PChar(URI),d);
end;

function TSablotEngine.SabParseStylesheetBuffer(const buffer:string;
        d:PDomDocument): Integer;
begin
  Result:=SablotParseStyleSheetBuffer(FSituation^,PChar(buffer),d);
end;

procedure TSablotEngine.SetBase(const scheme,base:string);
var
  i: Integer;
  s1, s2: string;
begin
  for i:=0 to FBases.count-1 do begin
   SplitKeyValue(FBases[i],s1,s2);
   if s1=scheme then begin
     FBases[i]:=scheme+'='+base;
     exit;
   end;
  end;
  if scheme<>'' then
    FBases.Add(scheme+'='+base)
end;

procedure TSablotEngine.SetDefaultBase(const ABase: string);
begin
  FDefaultBase:=ABase;
  SablotSetBase(FProcessor^,PChar(FDefaultBase));
end;

procedure TSablotEngine.SetParam(const paramName,paramValue:string);
var
  i: Integer;
  s1, s2: string;
begin
  for i:=0 to FParams.count-1 do begin
   SplitKeyValue(FParams[i],s1,s2);
   if s1=paramName then begin
     FParams[i]:=paramName+'='+paramvalue;
     exit;
   end;
  end;
  if paramName<>'' then
    FParams.Add(paramName+'='+paramvalue)
end;

{ TFileProxy }

{
********************************** TFileProxy **********************************
}
constructor TFileProxy.Create(Aowner:TComponent);
begin
  FHandles:=TList.Create;
  inherited;
end;

destructor TFileProxy.Destroy;
begin
  FHandles.Free;
  inherited;
end;

function TFileProxy.Close(handle:integer): Integer;
begin
  try
   FHandles.Remove(pointer(handle));
   TObject(handle).free;
   Result:=0;
  except
   Result:=1;
  end;
end;

function TFileProxy.FreeMemory(var buffer:PChar): Integer;
begin
  FInBuff:='';
  Result:=0;
end;

function TFileProxy.Get(const handle:integer;var buffer:PChar;var 
        bytecount:integer): Integer;
begin
  with TStream(handle) do
  try
   ReadBuffer(buffer,size);
   bytecount:=size;
   Result:=0;
  except
   Result:=0;
  end;
end;

function TFileProxy.GetAll(const scheme,rest:Pchar;var buffer:Pchar;var
        bytecount:integer): Integer;
var
  fname: string;
  FStream: TFileStream;
begin
  Result:=1;
  try
   fname:=Copy(rest,Pos('://',rest)+3,length(rest));
   if not FileExists(fname) then
    exit;
   FStream:=TFileStream.Create(fname,fmOpenRead);
   SetLength(FInBuff,fstream.size);
   FStream.ReadBuffer(pointer(FInBuff)^,Fstream.size);
   buffer:=Pchar(FInbuff);
   bytecount:=strlen(buffer);
   fstream.free;
   Result:=0;
  except
   if Assigned(FStream) then
    FStream.Free;
   FInBuff:='';
  end
end;

function TFileProxy.HasHandle(handle:integer): Boolean;
begin
  if  FHandles.IndexOf(Pointer(handle))<>-1 then
   Result:=true
  else
   Result:=false;
end;

function TFileProxy.Open(const scheme,rest:Pchar;var handle:integer): Integer;
var
  fname: string;
begin
  Result:=1;
  fname:=Copy(rest,Pos('://',rest)+3,length(rest));
  if FileExists(fname) then
   FOpenMode:=fmOpenwrite
  else
   FOpenMode:=fmCreate;
  FHandles.Add(TFileStream.Create(fname,FOpenMode));
  handle:=integer(FHandles[Fhandles.Count-1]);
  if handle<>0 then
   Result:=0;
end;

function TFileProxy.Put(handle:integer;const buffer:PChar;var 
        byteCount:integer): Integer;
begin
  try
  with TStream(handle) do begin
   WriteBuffer(buffer^,bytecount);
   bytecount:=bytecount+1;
   Result:=0;
  end;
  except
   Result:=1;
  end;
end;

end.




